unit GvTable;

{ Graphics Vision, Table support
  Copr. 1995, 1999 Matthias Koeppe

  $Id: gvtable.pas 1.4 1999/02/13 14:09:54 mkoeppe Exp $
}

{$ifdef FPK}
{$i fpkdef.pp}
{$else}
{$A+,B-,F+,G+,I-,O+,P-,Q-,R-,S-,T-,V+,X+}
{$endif}

interface

{$ifdef RWS}
uses Objects;
{$else}
uses Objects, Drivers, Views, WinRes, GvViews, GvDialog;
{$endif RWS}

type
{ Table column alignment
}
{$ifdef FPC}{$PACKENUM 1}{$endif}
  TAlignment = (LeftAlign, CenterAlign, RightAlign, DecimalAlign);
{$ifdef FPC}{$PACKENUM DEFAULT}{$endif}

{ Table column data
}
  PTableField = ^TTableField;
  TTableField = object(TObject)
    Title: PString;
    Alignment: TAlignment;
    Left, Right: Integer;
    Decimal: Integer;
    constructor Init(ATitle: string; AnAlignment: TAlignment;
      ALeft, ARight: Integer; ADecimal: Integer);
    constructor Load(var S: TStream);
    destructor Done; virtual;
    procedure Store(var S: TStream);
  end;

{ Table column collection
}
  PTableFields = ^TTableFields;
  TTableFields = object(TCollection)
    Position: Integer;
    Gap: Integer;
    constructor Init;
    procedure CenterField(ATitle: string; Width: Integer);
    procedure DecimalField(ATitle: string; Width: Integer; Decimal: Integer);
    procedure LeftField(ATitle: string; Width: Integer);
    procedure RightField(ATitle: string; Width: Integer);
  end;

{$ifndef RWS}
const
{ Extended list flags
}
  lfColoredTable = $0100;

type
{ Generic table
}
  PTableViewer = ^TTableViewer;
  TTableViewer = object(TListBox)
    HScrollbar: PScrollbar;
    HDelta: Integer;
    TableFields: PCollection;
    constructor Init(var Bounds: TRect; AHScrollbar, AVScrollbar: PScrollbar;
      ATableFields: PCollection);
    constructor Load(var S: TStream);
    destructor Done; virtual;
    procedure Awaken; virtual;
    procedure ChangeBounds(var Bounds: TRect); virtual;
    procedure Draw; virtual;
    procedure DrawItem(Item: Integer); virtual;
    procedure DrawItemText(Item: Integer; R: TRect); virtual;
    procedure DrawTitle; virtual;
    procedure DrawVertLines; virtual;
    procedure GetItemRect(Item: Integer; var R: TRect); virtual;
    procedure GetItemSubRect(var R: TRect); virtual;
    procedure GetTitleRect(var R: TRect); virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure Store(var S: TStream);
  private
    procedure CalcHBar;
  end;

{ Table item, providing textual representation
}
  PTableItem = ^TTableItem;
  TTableItem = object(TObject)
    function GetText(Info: LongInt): string; virtual;
  end;

{ Table of TTableItem objects
}
  PTable = ^TTable;
  TTable = object(TTableViewer)
    Info: LongInt;
    constructor Init(var Bounds: TRect; AHScrollbar, AVScrollbar: PScrollbar;
      ATableFields: PCollection; AnInfo: LongInt);
    constructor Load(var S: TStream);
    procedure GetInfo(var S: TStream); virtual;
    function GetText(Item: Integer; MaxLen: Integer): string; virtual;
    procedure PutInfo(var S: TStream); virtual;
    procedure Store(var S: TStream);
  end;

const
  { Stream registration records
  }
  RTableField: TStreamRec = (
    ObjType: 250;
    VmtLink: Ofs(TypeOf(TTableField)^);
    Load:    @TTableField.Load;
    Store:   @TTableField.Store);

  RTableFields: TStreamRec = (
    ObjType: 251;
    VmtLink: Ofs(TypeOf(TTableFields)^);
    Load:    @TTableFields.Load;
    Store:   @TTableFields.Store);

  RTableViewer: TStreamRec = (
    ObjType: 252;
    VmtLink: Ofs(TypeOf(TTableViewer)^);
    Load:    @TTableViewer.Load;
    Store:   @TTableViewer.Store);

  RTable: TStreamRec = (
    ObjType: 253;
    VmtLink: Ofs(TypeOf(TTable)^);
    Load:    @TTable.Load;
    Store:   @TTable.Store);

{ Class init procedure
}
procedure InitTable(Data: PControlData; Info: PDialogInfo);

const
  { Class registration record
  }
  classTable: TClassRec = (
    ClassId: 'GvTable';
    Init:  @InitTable);

{ Stream and class registration procedure
}
procedure RegisterGvTable;

{$endif RWS}

{$ifdef FPC}
{$PACKRECORDS 1}
{$endif}
type
{ Table field, binary representation
}
  PTableFieldB = ^TTableFieldB;
  TTableFieldB = record
    Alignment: TAlignment;
    Decimal: Byte;
    Width: Integer;
    Title: string[1];
  end;

  PTableB = ^TTableB;
  TTableB = record
    Count: Byte;
    Fields: record end;
  end;
{$ifdef FPC}
{$PACKRECORDS DEFAULT}
{$endif}

const
  ts_Yellow = 1;

function CreateTableFields(var TableB; BaseUnits: Integer): PTableFields;

{$ifdef RWS}
function CreateTableFieldB(Fields: PTableFields; var TableB; BaseUnits: Integer): Word;
{$endif RWS}

implementation

{$ifdef FPC}
uses Bgi, ExtGraph, GvWinDlg;
{$else}
{$ifndef RWS}
{$ifdef Windows}
uses WinGr, ExtGraph, GvWinDlg;
{$else}
uses MetaGr, MyFonts, ExtGraph, GvWinDlg;
{$endif Windows}
{$endif RWS}
{$endif}

{  TTableField
}

constructor TTableField.Init(ATitle: string; AnAlignment: TAlignment;
  ALeft, ARight: Integer; ADecimal: Integer);
begin
  inherited Init;
  Title := NewStr(ATitle);
  Alignment := AnAlignment;
  Left := ALeft;
  Right := ARight;
  Decimal := ADecimal;
end;

constructor TTableField.Load(var S: TStream);
begin
  Title := S.ReadStr;
  S.Read(Alignment, SizeOf(Alignment) + 3 * SizeOf(Integer))
end;

destructor TTableField.Done;
begin
  DisposeStr(Title);
  inherited Done
end;

procedure TTableField.Store(var S: TStream);
begin
  S.WriteStr(Title);
  S.Write(Alignment, SizeOf(Alignment) + 3 * SizeOf(Integer))
end;

{  TTableFields
}

constructor TTableFields.Init;
begin
  inherited Init(8, 8);
  Position := 0;
  Gap := 3
end;

procedure TTableFields.CenterField(ATitle: string; Width: Integer);
begin
  Insert(New(PTableField, Init(ATitle, CenterAlign,
    Position, Position + Width, 0)));
  Inc(Position, Width + Gap)
end;

procedure TTableFields.DecimalField(ATitle: string;
  Width: Integer; Decimal: Integer);
begin
  Insert(New(PTableField, Init(ATitle, DecimalAlign,
    Position, Position + Width, Decimal)));
  Inc(Position, Width + Gap)
end;

procedure TTableFields.LeftField(ATitle: string; Width: Integer);
begin
  Insert(New(PTableField, Init(ATitle, LeftAlign,
    Position, Position + Width, 0)));
  Inc(Position, Width + Gap)
end;

procedure TTableFields.RightField(ATitle: string; Width: Integer);
begin
  Insert(New(PTableField, Init(ATitle, RightAlign,
    Position, Position + Width, 0)));
  Inc(Position, Width + Gap)
end;

{$ifndef RWS}

{  TTableViewer
}

constructor TTableViewer.Init(var Bounds: TRect; AHScrollbar, AVScrollbar: PScrollbar;
  ATableFields: PCollection);
begin
  inherited Init(Bounds, AVScrollbar);
  HScrollbar := AHScrollbar;
  GrowMode := gfGrowHiX + gfGrowHiY;
  Flags := Flags or lfColoredTable;
  {with HScrollbar^ do
    Flags := Flags or sbHandleKeyboard;}
  TableFields := ATableFields;
  CalcHBar
end;

constructor TTableViewer.Load(var S: TStream);
begin
  inherited Load(S);
  GetPeerViewPtr(S, HScrollbar);
  TableFields := PCollection(S.Get);
end;

destructor TTableViewer.Done;
begin
  If TableFields <> nil
  then Dispose(TableFields, Done);
  inherited Done
end;

procedure TTableViewer.Awaken;
begin
  CalcHBar
end;

procedure TTableViewer.CalcHBar;
var
  Min, Max: Integer;
  R: TRect;

	procedure MinMax(Field: PTableField); {$ifndef FPK}far;{$endif}
	begin
	  with Field^ do
	  begin
	    If Left < Min then Min := Left;
	    If Right > Max then Max := Right
	  end
	end;

begin
  Min := MaxInt;
  Max := - MaxInt;
  TableFields^.ForEach(@MinMax);
  GetItemSubRect(R);
  If HScrollbar <> nil
  then begin
    HScrollbar^.SetRange(Min, Max - (R.B.X - R.A.X));
    HScrollbar^.SetStep((Size.X * 2) div 3, Size.X div 10)
  end
end;

procedure TTableViewer.ChangeBounds(var Bounds: TRect);
begin
  inherited ChangeBounds(Bounds);
  CalcHBar
end;

procedure TTableViewer.Draw;
var
  R: TRect;
begin
  inherited Draw;
  DrawTitle;
  GetExtent(R);
  SetSubRect(R);
  GetTitleRect(R);
  SetColor(0);
  HoriLine(0, R.B.Y + 1, Size.X - 1);
  DrawVertLines
end;

procedure TTableViewer.DrawItem(Item: Integer);
var
  R, Sub: TRect;

	function Yellow: Boolean;
	begin
	  Yellow := Odd(Item)
	end;

begin
  If Item>=Range then Exit;
  If (Flags and lfColoredTable = 0) or (Item = Focused)
  then inherited DrawItem(Item)
  else begin
    SetViewPort;
    GetItemRect(Item, R);
    GetItemSubRect(Sub);
    Sub.Intersect(R);
    If SetSubRect(Sub)
    then begin
      If Yellow
      then begin
	{ This approach doesn't work properly, since
	  there occur pattern faults in the Bar routine...
	}
	(*SetFillStyle(SolidFill, 7);
	Bar (R.A.X, R.A.Y, R.B.X-1, R.B.Y-1);
	SetFillStyle(CloseDotFill, 1);
	SetWriteMode(OrPut);
	Bar (R.A.X, R.A.Y, R.B.X-1, R.B.Y-1);
	SetWriteMode(NormalPut)*)

	{ The new approach
	}
	ColorFill(- HDelta, R.A.Y, Sub.B.X, Sub.B.Y, 15, 14, @CloseDot);
      end
      else begin
	SetFillStyle(SolidFill, 15);
	Bar (R.A.X, R.A.Y, R.B.X-1, R.B.Y-1)
      end;
      SetTextJustify(LeftText, CenterText);
      SetTextParams(ftSansSerif, 0, GetColor(3), false);
      DrawItemText(Item, R);
    end;
    RestoreViewPort
  end
end;

procedure TTableViewer.DrawItemText(Item: Integer; R: TRect);
var
  s: string;
  yCenter: Integer;
  xDelta: Integer;

	procedure DrawField(Field: PTableField); {$ifndef FPK}far;{$endif}
	var
	  t, d: string;
	  i, w: Integer;
	  sub: TRect;
	begin
	  i := Pos(#9, s);
	  If i = 0 then i := Length(s) + 1;
	  t := Copy(s, 1, i - 1);
	  Delete(s, 1, i);
	  If t <> '' then
	  with Field^ do
	  begin
	    sub := R;
	    If sub.A.X < Left + xDelta then sub.A.X := Left + xDelta;
	    If sub.B.X > Right + xDelta then sub.B.X := Right + xDelta;
	    If SetSubRect(sub) then
	    case Alignment of
	      LeftAlign:
		begin
		  SetTextJustify(LeftText, CenterText);
		  OutTextXY(Left + xDelta, yCenter, t)
		end;
	      RightAlign:
		begin
		  SetTextJustify(RightText, CenterText);
		  OutTextXY(Right + xDelta - 1, yCenter, t)
		end;
	      CenterAlign:
		begin
		  SetTextJustify(CenterText, CenterText);
		  OutTextXY((Left + Right) div 2 + xDelta, yCenter, t)
		end;
	      DecimalAlign:
		begin
		  i := Pos('.', t);
		  w := Right - (Decimal * TextWidth('0') + TextWidth('.')) + xDelta + 1;
		  SetTextJustify(RightText, CenterText);
		  If i = 0
		  then OutTextXY(w, yCenter, t)
		  else begin
		    d := Copy(t, i, Length(t) - i + 1);
		    t[0] := Chr(i - 1);
		    OutTextXY(w, yCenter, t);
		    SetTextJustify(LeftText, CenterText);
		    OutTextXY(w + 1, yCenter, d)
		  end
		end;
	    end
	  end
	end;

begin
  DrawVertLines;
  xDelta := R.A.X - HDelta;
  yCenter := (R.B.Y + R.A.Y) div 2;
  s := GetText(Item, 255);
  TableFields^.ForEach(@DrawField);
end;

procedure TTableViewer.DrawTitle;
var
  R: TRect;
  xDelta: Integer;

	procedure DrawField(Field: PTableField); {$ifndef FPK}far;{$endif}
	var
	  sub: TRect;
	begin
	  with Field^ do
	    If Title <> nil
	    then begin
	      sub := R;
	      If sub.A.X < Left + xDelta then sub.A.X := Left + xDelta;
	      If sub.B.X > Right + xDelta then sub.B.X := Right + xDelta;
	      If SetSubRect(sub)
	      then begin
		SetTextJustify(CenterText, CenterText);
		OutTextXY((Left + Right) div 2 + xDelta,
		  (R.A.Y + R.B.Y) div 2, Title^)
	      end
	    end
	end;

begin
  GetTitleRect(R);
  xDelta := R.A.X - HDelta;
  SetTextParams(ftSansSerif, 1, 0, true);
  If SetSubRect(R)
  then TableFields^.ForEach(@DrawField)
end;

procedure TTableViewer.DrawVertLines;
var
  i: Integer;
  x, xDelta: Integer;
  R: TRect;
begin
  GetItemSubRect(R);
  xDelta := R.A.X - HDelta;
  with TableFields^ do
  For i := 0 to Count - 2 do
  begin
    x := (PTableField(At(i))^.Right + PTableField(At(i + 1))^.Left) div 2 + xDelta;
    SetColor(0);
    VertLine(x, 0, Size.Y - 1);
    SetColor(15);
    VertLine(x - 1, R.A.Y, Size.Y - 2);
    VertLine(x + 1, R.A.Y, Size.Y - 2);
  end
end;

procedure TTableViewer.GetItemRect(Item: Integer; var R: TRect);
begin
  GetItemSubRect(R);
  R.B.Y := R.A.Y + 15;
  R.Move(0, (Item-TopItem) * 15)
end;

procedure TTableViewer.GetItemSubRect(var R: TRect);
begin
  R.Assign(2, 21, Size.X - 2, Size.Y - 2);
end;

procedure TTableViewer.GetTitleRect(var R: TRect);
begin
  R.Assign(2, 2, Size.X - 2, 18)
end;

procedure TTableViewer.HandleEvent(var Event: TEvent);
var
  R: TRect;
  Delta: TPoint;
begin
  If (Event.What = evKeydown) and (HScrollbar <> nil)
  then HScrollbar^.HandleEvent(Event);
  inherited HandleEvent(Event);
  If (Event.What = evBroadcast) and (Event.Command = cmScrollbarChanged) and
     (Event.InfoPtr = HScrollbar) and (HScrollbar <> nil)
  then begin
    If not GetState (sfSelected) then Select;
    GetTitleRect(R);
    R.A.Y := 1;
    R.B.Y := Size.Y - 1;
    Delta.X := HDelta - HScrollbar^.Value;
    Delta.Y := 0;
    HDelta := HScrollbar^.Value;
    ScrollView(@Self, Delta, @R);
  end
end;

procedure TTableViewer.Store(var S: TStream);
begin
  inherited Store(S);
  PutPeerViewPtr(S, HScrollbar);
  S.Put(TableFields)
end;

{  TTableItem
}

function TTableItem.GetText(Info: LongInt): string;
begin
  Abstract
end;

{  TTable
}

constructor TTable.Init(var Bounds: TRect; AHScrollbar, AVScrollbar: PScrollbar;
  ATableFields: PCollection; AnInfo: LongInt);
begin
  inherited Init(Bounds, AHScrollbar, AVScrollbar, ATableFields);
  Info := AnInfo
end;

constructor TTable.Load(var S: TStream);
begin
  inherited Load(S);
  GetInfo(S)
end;

procedure TTable.GetInfo(var S: TStream);
begin
  S.Read(Info, SizeOf(Info))
end;

function TTable.GetText(Item: Integer; MaxLen: Integer): string;
begin
  If Item < List^.Count
  then GetText := PTableItem(List^.At(Item))^.GetText(Info)
end;

procedure TTable.PutInfo(var S: TStream);
begin
  S.Write(Info, SizeOf(Info))
end;

procedure TTable.Store(var S: TStream);
begin
  inherited Store(S);
  PutInfo(S)
end;

{ 
}

procedure InitTable(Data: PControlData; Info: PDialogInfo);
var
  R, SBRH, SBRV: TRect;
  SBH, SBV: PScrollbar;
  P: PGView;
  Inf: LongInt;
  E: Word;
  Fields: PTableFields;
begin
  GetControlRect(R, Data, Info);
  with Data^, Info^ do
  begin
    Group := Group or (lStyle and ws_Group <> 0);
    If bExtraSize = 0
    then Fields := New(PTableFields, Init)
    else Fields := CreateTableFields(bExtra, 4);
    SplitRectSB(R, SBRH, SBRV, lStyle and ws_HScroll <> 0, lStyle and ws_VScroll <> 0);
    If lStyle and ws_HScroll <> 0
    then begin
      SBH := New(PScrollBar, Init(SBRH));
      InsertControl(Dialog, SBH)
    end
    else SBH := nil;
    If lStyle and ws_VScroll <> 0
    then begin
      SBV := New(PScrollBar, Init(SBRV));
      InsertControl(Dialog, SBV)
    end
    else SBV := nil;
    If Seg(szText^) = 0
    then Inf := Ofs(szText^)
    else Val(szText^, Inf, E);
    P := New(PTable, Init(R, SBH, SBV, Fields, Inf));
    If lStyle and ts_Yellow <> 0 then
    with PTable(P)^ do
      Flags := Flags or lfColoredTable;
    P^.HelpCtx := wId;
    InsertControl(Dialog, P);
    Group := false;
    CreateLinks(Info, P)
  end
end;

procedure RegisterGvTable;
begin
  RegisterType(RTableField);
  RegisterType(RTableFields);
  RegisterType(RTableViewer);
  RegisterType(RTable);
  RegisterClass(classTable);
end;

{$endif RWS}

function CreateTableFields(var TableB; BaseUnits: Integer): PTableFields;
{ Binary -> Collection. }
var
  P: PTableFieldB;
  i: Integer;
  Fields: PTableFields;
  Field: PTableField;
  w: Integer;
begin
  Fields := New(PTableFields, Init);
  P := @TTableB(TableB).Fields;
  For i := 1 to TTableB(TableB).Count do
  begin
    w := (P^.Width * BaseUnits + 2) div 4;
    {$ifndef Windows}
    AnsiTo437Str(@P^.Title);
    {$endif Windows}
    Field := New(PTableField, Init(P^.Title, P^.Alignment,
      Fields^.Position, Fields^.Position + w, P^.Decimal));
    Fields^.Insert(Field);
    Inc(Fields^.Position, w + Fields^.Gap);
    Inc(PChar(P), SizeOf(TTableFieldB) + Length(P^.Title) - 1);
  end;
  CreateTableFields := Fields
end;

{$ifdef RWS}
function CreateTableFieldB(Fields: PTableFields; var TableB; BaseUnits: Integer): Word;
{ Collection -> binary. Size is returned. }
var
  P: PTableFieldB;
  i: Integer;
  Field: PTableField;
begin
  TTableB(TableB).Count := Fields^.Count;
  P := @TTableB(TableB).Fields;
  For i := 0 to Fields^.Count - 1 do
  begin
    Field := Fields^.At(i);
    P^.Alignment := Field^.Alignment;
    P^.Width := ((Field^.Right - Field^.Left) * 4) div BaseUnits;
    P^.Decimal := Field^.Decimal;
    If Field^.Title = nil
    then P^.Title[0] := #0
    else Move(Field^.Title^, P^.Title, Length(Field^.Title^) + 1);
    Inc(PtrRec(P).Ofs, SizeOf(TTableFieldB) + Length(P^.Title) - 1);
  end;
  CreateTableFieldB := Ofs(P^) - Ofs(TableB);
end;
{$endif RWS}

end.
