#This file is intended to allow users to evaluate and experiemnt with Wheel Polynomials as well as perform operations related to Young Diagrams
#Things created in this file:
#(class)DyckWord with attribute: word(string). Has functions to modify word as well as give permutation representation of the Dyck Word.
#For relevent methods associated with Dyck Word Objects, see documentation bellow.
#AllDyckWords(n): returns a list of all DyckPath Objects in NC_n
#DrawMatching: Takes permutation and returns arc matching diagram
#SortMatch: Takes non-crossing matching and returns one with alphabetical/increasing ordering [[3,2],[1,4]]-> [[1,4],[2,3]]
#LO(Perm, j): j-th Temporary Lieb operator. Takes NCPerm as argument.
#MatchToDw: Takes Non-crossing Matching and returns the corresponding Dyck Word (String)(ex [[1,2]] -> '()')
#LO_PreImage(n,j,NC): calculates the pre-image of a non crossing permutation NC(in NC_n) under the Temporary Lieb Operator e_j
#CreateMinp(n): Takes in n and returns Wheel Polynomial corresponding to NC_n minimal matching.
#minp=CreateMinp(n) for the n specified by the user at runtime(This is generated on each run for the sake of reducing runtime for more complicated operations).
#IsWheelPoly(poly): Takes polynomial and return true if it is a wheel polynomial and false otherwise.
#DisplayWPoly(poly): Takes a wheel polynomial and displays it in a more notationally appealing way.
#skf(poly): Takes Wheel poly and swapps z_k and z_{k+1}
#dkf(poly): Applies k-th 'differential operator to poly
#Qkz(dw): Takes a DyckWord(as defined above) and returns the qkz basis polynomial corresponding to that DyckWord
#Wheel_Z_Eval(poly): Takes wheel polynomial, poly, and evaluates poly where zi=1 for all i
#Wheel_P_Eval(poly): Takes wheel polynomial, poly, and evaluates poly with the given permuation evaluation: zi=q if perm[i]\leq i, zi=q^2 if perm[i]>i
#Qkz_Basis(): returns the a list of the qkz basis elements for the n input value specified by the user. It is advised that users should set output to write to a designated txt file instead of display in the terminal due to the large size of Qkz Basis Polynomials.
#Qkz_Decomp(poly): Takes a wheel polynomial, poly, and returns a list of the qkz basis coefficiencts in the same order as the corresponding Dyck Words in the pre-generated list AllDW(See documentation at end of document).
#Qkz_Poly(dw): Intended to be used with Qkz and global list QkzBasis. Takes Dyck Word of length 2n as input and returns the corresponding Qkz basis element from QkzBasis.


#This package is incorporated into Qkz_Basis() to allow the user to take advantage of multiple threads on their processor instead of just the default 1 when running Qkz_Basis()

from multiprocessing import Pool

#This line allows the user to specify which value of n corresponding to NC_n they are working in. It is worth noting that all of the methods in this file can be easily modified to work independently should the user prefer this.We have 'restricted' n-values to be greater than 1 due to the fact that the cases for n=1 all wheel polynomials are constants. This causes methods involving polynomial evaluation to throw an error unless exceptions are written in.

n=input('Please enter the desired n-value(n>1): ')
while(n<= 1):
    print 'Invalid Input'
    n= input('Please enter the desired n-value(n>1): ')

#This is the DyckWord class. It has one private variable 'word' wich stores the corresponding Dyck Word as a string.
class DyckWord:
    word='('
    
    def __init__(self, word):
        self.word=word

    #Takes a string as input and sets the dych word to that string.
    def SetWord(self, M):
        self.word=M
        
    def addLeft(self):
        self.word=self.word + '('
        
    #Adds right  parenthasis to word. 
    def addRight(self):
        self.word=self.word + ')'
        
    #Adds left parenthasis to word.
    def Left(self):
        return self.word.count('(')
    
    #Returns number of right parenthasis.
    def Right(self):
        return self.word.count(')')
    
    #Returns the number of left parenthasis
    def Word(self):
        return self.word
    
    # Prints the Dyckword as well as length of the word.
    def Display(self):
        print self.word+'    ' + 'Length: ' +  str(len(self.word))  
    
    #Returns the non-crossing matching that corresponds to the Dyck Word
    def Matching(self):
        Matching=[]
        RightBrack=[]
        Matched=[]
        for i in range(len(self.word)):
            if self.word[i]==')':
                RightBrack.append(i)
        for i in RightBrack:
            for j in reversed(range(i)):
                if self.word[j]=='(' and (j not in Matched):
                    Matching.append([j+1,i+1])
                    Matched.append(i)
                    Matched.append(j)
                    break
                        
        return SortMatch(Matching)
    
    #Returns the NC_n^+ sequence(as a list) that corresponds to the DyckWord. Th
    def Seq(self):
        M=self.Matching()
        s=[]
        for i in M:
            s.append(i[0])
        return s
    
#Takes a DyckWord 'other' as well as a positive integer k and uses the NC_n^+ sequence from above to determine if the young diagram of 'other' can be obtained from the young diagram of this Dyckword by the addition of k blocks.       
    def Is_k_Bellow(self, other,k):
        b=other.Seq()
        a=self.Seq()
        s=0
        for i in range(n):
            diff=b[i]-a[i]
            if diff<0:
                return False
            else:
                s+= diff
        if (s==k and k!=(-1)) or (s>0 and k==(-1)) :
            return True
        else:
            return False
    #Returns the permutation representation of this DyckWord(as a list)   
    def Permutation(self):
        Permutation=range(2*n)
        for i in self.Matching():
            Permutation[i[0]-1]=i[1]
            Permutation[i[1]-1]=i[0]
        return Permutation
        
#Takes input n(corresponding to NC_n) and returns a list of all DyckPath objects corresponding to NC_n. This algorithm works by starting with an initial word '(' and then growing that word by adding a '(' or a ')' based on the rules for DyckPaths. The first few iterations are as follows: ['('], ['((', '()'], ['(((','()(', '(()'].
def AllDyckWords(n):    
    NC=[]
    NC.append(DyckWord('('))

    for i in range(2*n-1):
        for i in range(len(NC)):
            if NC[i].Left()==NC[i].Right():
                NC[i].addLeft()
            elif NC[i].Left()>NC[i].Right():
                if NC[i].Left()==n:
                    NC[i].addRight()
                else:
                    b=DyckWord(NC[i].Word())
                    b.addRight()
                    NC.append(b)
                    NC[i].addLeft()
    
    
    return sorted(NC,key=lambda x: sum(x.Seq()))              

#Takes matching b as input and returns a Graphics oject that represents the non-crossing diagram for that matching.
def DrawMatching(b):
    command='arc((%s,0), %s, sector=(0,pi))'%(N((b[0][1]+b[0][0])/2,digits=5), N(abs((b[0][1]-b[0][0]))/2,digits=5))
    for i in range(1,len(b)):
        command=command + '+ arc((%s,0), %s, sector=(0,pi))'%(N((b[i][1]+b[i][0])/2,digits=5), N(abs(b[i][1]-b[i][0])/2,digits=5))
    return eval(command)

def SortMatch(P):
    for i in P:
        if i[0]>i[1]:
            t1=i[0]
            t2=i[1]
            i[1]=t1
            i[0]=t2
    return sorted(P,key=lambda x:x[0])
    
# Takes a Matching P and an index j and returns e_j(P), where e_j is the j-th Lieb Temporary operator.    
def LO(P,j):
    k1=j
    k2=j+1
    if j==(2*n):
        k2=1
    np=[]
    temp=[]
    for i in P:
        if (k1  not in i) and (k2 not in i):
            np.append(i)
        else:
            if i[0]==k2:
                temp.append(i[1])
            elif i[1]==k2:
                temp.append(i[0])
            if i[0]==k1:
                temp.append(i[1])
            elif i[1]==k1:
                temp.append(i[0])
    np.append(temp)
    if [k1,k2] not in np and [k2,k1] not in np:
        np.append([k1,k2])
    
    return SortMatch(np)

#PermToDP takes a permutation/matching(perfect) and returns the corresponding DyckWord(Type: string) representation.
def MatchToDw(P):
        ma=range(2*len(P))
        word=''
        for i in P:
            ma[i[0]-1]='('
            ma[i[1]-1]=')'
        for i in ma:
            word=word+i
        return word
    
#This operator takes an n value corresponding to NC_n, a j value corresponding to e_j, and a non-crossing permuation/matching. It returns a list of all DyckPaths that are in e_j^{-1}(NC).
def LO_PreImage(n,j, NC):
    P=[]
    for i in AllDW:
        if LO(i.Matching(), j)==NC:
            P.append(i)
    return P

#It is important to not that the ring QQ was used instead of CC because combining CC with a cyclotomic polynomial ring produced only approximate complex outputs. This made evaluations comversome and visual output unappealing.
R = PolynomialRing(QQ, (2*n)+1 , 'z')
z=R.gens()

# I found this ring to be better to work over as it produced much cleaner outputs. Here q is defined to by the cubed root of unity.
cyc=CyclotomicField(3)
l=cyc.gens()
q=l[0]


# Minp is the default wheel polynomial stated in the intro to section 2 of Dan Romik's paper. Minp corresponds to the Wheel Polynomial that represents the minimal non-crossing matching.
def CreateMinp(k):
    Minp=1
    for i in range (1,k+1):
        for j in range(i+1, k+1):
            Minp=(Minp*(q*z[i]-((q)^2)*z[j])*(q*z[n +i]-(q^2)*z[n+j]))
    return ((-3)^((-1)*binomial(n,2)))*Minp    


#This method tests if a polynomial f is a wheel polynomial. Still need to add homogeneous verification.(Asssumed n>1)
def IsWheelPoly(poly):
    if poly==0:
        return False
    for i in range(1,2*n+1):
        for j in range(i+1,2*n+1):
            for k in range(j+1,2*n):
                g1=eval('poly(z%s=q*z[%s])'%(j,k))
                g=eval('g1(z%s=(q**2)*z[%s])'%(i,k))
                g
                if g!=0:
                    return False
    # The following steps check homogeneity.
    g=poly
    for i in range (1,2*n+1):
        g=eval('g(z%s=z[0])'%i)
    
    t=g/g(z0=1)
    if 2^((n^2-n)) != t(z0=2):
        return False
        
    return True

# This function just gives a cleaner presentation of a wheel polynomial. We have fixed the variable indexing to match the standard indexing i->i+1. DisplayWPoly just takeas a wheel polynomial as an input.
def DisplayWPoly(poly):
    if poly != 1:
        poly=str(poly).replace('(zeta3)', 'q')
        poly=poly.replace('(-zeta3 - 1)', '(q^2)')
        poly=poly.replace('(zeta3 + 1)', '(-q^2)')
        poly=poly.replace('zeta3', 'q')
    print poly

# This operation just takes a polynomial input (poly)  and swaps z_k and z_{k+1} .
def skf(poly, k):
    if k > 2*n-1:
        print 'k too large'
        return
    
    else:
        ph=z[0]
    
        g1=eval('poly(z%s=ph, z%s=z[%s])'%(k,(k+1),k))
        g2=eval('g1(z%s=z[%s])'%(0,k+1))
        return g2
    
#This is the differential operator defined in Def 2.1 [Aigner]. It takes a wheel polynomial and an index k as input and returns D_k(poly). We use the quo_rem() function to do polynomial division instead of sage's built in feature so that our output does not get converted to a fractional field type. The fractional field type is not compatible with fast_callable() which is used bellow to evaluate wheel polynomials.
def dkf(poly, k):
    if k > 2*n-1:
        print 'k too large'
        return
    
    return ((q*z[k]-(q^2)*z[k+1])*(skf(poly,k)-poly)).quo_rem((z[k+1]-z[k]))[0]
    #return (q*z[k]-(q^2)*z[k+1])*(skf(poly,k)-poly)/(z[k+1]-z[k])
        
def Qkz(dw):
    #if dw.Word()==AllDW[0].Word():
        #return minp
    Q=Qkz_Poly(dw)
    if Q!='NULL':
        return Q
    else:
        j=0
        sigma=0
        s=0
        L0=[]
        M=dw.Matching()
        for i in AllDW:
            if i.Is_k_Bellow(dw,1):
                L0.append(i)
        for i in range(1,2*n):
            for k in L0:
                L1=LO_PreImage(n,i,k.Matching())
                for m in L1:
                    if m.Word()==dw.Word():
                        sigma=k
                        j=i
 
        L2=LO_PreImage(n,j,sigma.Matching())
        for i in L2:
            if i.Word()==sigma.Word():
                L2.remove(i)
        for i in L2:
            if i.Word()==dw.Word():
                L2.remove(i)
        for i in L2:
            s+=Qkz(i)
        return dkf(Qkz(sigma), j)- s

#This method evaluates a wheel polynomial at 1 i.e. zi=1 for all i.
def Wheel_Z_Eval(poly):
    for i in range(2*n):
        poly=eval('poly(z%s=1)'%i)
    return poly

#This method evaluates a wheel polynomial, poly, with the given permuation evaluation: zi=q if perm[i]\leq i, zi=q^2 if perm[i]>i.
def Wheel_P_Eval(poly, perm):
    evaluation=[]
    for i in range(2*n):
        if perm[i]>(i+1):
            evaluation.append(1)
        else:
            evaluation.append(-1)
    s='1'
    for i in range(2*n):
        s+= ',q**(%s)'%(-evaluation[i])
    poly=fast_callable(poly)
    return eval('poly(%s)'%s)

    #Bellow is the old code for evaluating. By using the fast_callable method, I noticed marginal improvements in speed.
    #for i in range(2*n):
        #poly=eval('poly(z%s=q**(%s))'%(i+1,((-1)*evaluation[i])))
    #return poly



#This method takes a wheel polynomial and returns a list of the coefficients for its qkz basis decomposition. The order of the coefficients directly corresponds to the ording of dych words in AllDW.
def Qkz_Decomp(poly):
    Decomp=[]
    for i in AllDW:
        Decomp.append(Wheel_P_Eval(poly, i.Permutation()))
    return Decomp

#To reduce rpeated compuation we pre-generate all possible DyckWords in the list AllDW. AllDW is called on directly in the majority of the above functions. Fortunately generating this list is not very computationionally intesive. Users can use this pre-generated list when writing scripts bellow. We have also pre-generated the 'minimal' qkz polynomial corresponding to ()_n. Omitting minp line can drastically improve computational time if the user does not wish to compute wheel polynomials. We note that Qkz() and Qkz_Basis() are the only function that call the variable minp bellow directly. 

AllDW=AllDyckWords(n)

#The following sets up an array to store generated Qkz Polynomials. This speeds up recursive calculations by preventing repeated work.
QkzBasis=[]
for i in AllDW:
    QkzBasis.append('NULL')
QkzBasis[0]=CreateMinp(n)


#This method takes a Dyck Word of length 2n and returns the corresponding Qkz basis element from QkzBasis. Returns "Null" if that element has not been generated yet. In this case, refer to the Qkz method.

def Qkz_Poly(dw):
    for i in range(len(AllDW)):
        if dw.Word()== AllDW[i].Word():
            return QkzBasis[i]


#Function designated to run for each thread in Qkz_Basis()        
def dothis(i):
    return Qkz(AllDW[i])
           
    

#Generates Qkz Polynomials in order of number of block in corresponding Young Diagrams.
def Qkz_Basis():
    
    blocks=[]
    current=0
    #Gives list of indices for when block number changes.
    for i in range(len(AllDW)):
        k=sum(AllDW[i].Seq())
        if k != current:
            blocks.append(i)
            current=k
    blocks.append(len(AllDW))

    #Instead of running as many threads as possible concurrently, this method complets all Qkz Polynomials for a given block number before moving on to a larger block number. This prevents duplicated work from occurring in the recursive method Qkz().
    for i in range(1,len(blocks)-1):    
        if __name__ == '__main__':
            # Pool() takes the number threads to be used as an argument. Users should adjust this  to suit there system limitations.
            p=Pool(threads)
            new=(p.map(dothis, range(len(AllDW))[blocks[i]:blocks[i+1]]))
	    for j in range(len(new)):
	        QkzBasis[blocks[i]+j]=new[j]
            p.terminate()

    return QkzBasis

#Bellow is some sample code:

threads=3 #number of threads to be used.
Qkz_Basis()


for i in range(len(AllDW)):
    print '\n'
    print i
    print AllDW[i].Word()
    print AllDW[i].Matching()
    print AllDW[i].Permutation()
    print AllDW[i].Seq()
    DrawMatching(AllDW[i].Matching()) #Will only display in a sage worksheet/IDE








