// the following functions implement the parameterization of the family
// of ellipses which pass through the following two edges.
//      position (x_i,y_i), slope -p_i/q_i

//
// lambda_range() produces the value of lambda_0. The interval
// [0,lambda_0] is thus the domain of the parameterization.
//
float lambda_range(float x1,float y1,float p1,float q1,
              float x2,float y2,float p2,float q2)
{
float num0 = 4*(p1*(x2-x1)+q1*(y2-y1))*(p2*(x1-x2)+q2*(y1-y2));
float den0 = (p1*q2-p2*q1)*(p1*q2-p2*q1);
float lam0 = num0/den0;
return (lam0);
}

//
// intermed_calc() inputs the edge info and a value of lambda
// (in the range determined above) and produces the intermediate
// results a,b,c,f,g,h from which all the ellipse's parameters
// can be determined
//
void intermed_calc(float x1,float y1,float p1,float q1,
              float x2,float y2,float p2,float q2,float lambda,
              float *a,float *b,float *c,float *f,float *g,float *h)
{
float y1y2=y1-y2;
float y1y2s=y1y2*y1y2;
float x2x1=x2-x1;
float x2x1s=x2x1*x2x1;
float xy12=x1*y2-x2*y1;
float xy12s=xy12*xy12;
float p1p2=p1*p2;
float q1q2=q1*q2;
float pq12=p1*q2+p2*q1;
float px1=p1*x1+q1*y1;
float px2=p2*x2+q2*y2;
*a=y1y2s-lambda*p1p2;
*b=x2x1s-lambda*q1q2;
*c=xy12s-lambda*px1*px2;
*f=xy12*x2x1+0.5*lambda*(q1*px2+q2*px1);
*g=xy12*y1y2+0.5*lambda*(p1*px2+p2*px1);
*h=y1y2*x2x1-0.5*lambda*pq12;
}

//
// center() takes as input the values a,b,..,h and produces the
// center position of the corresponding ellipse
//
void center(float *x0,float *y0,float a,float b,float c,float f,float g,float h)
{
float detnt=a*b-h*h;

*y0=(h*g-a*f)/detnt;
*x0=(h*f-b*g)/detnt;
}

//
// area() takes as input the values a,b,..,h and produces the
// area of the corresponding ellipse
//
float area(float a,float b,float c,float f,float g,float h)
{
float detnt=a*b-h*h;
float num2=a*f*f-2*h*f*g+b*g*g;
float d=num2/detnt-c;
if(PI*d/sqrt(detnt) > 0.0)
     return (PI*d/sqrt(detnt));
else
     return (-PI*d/sqrt(detnt));
}


//
// eigen_value returns the eigenvalues of a 2 by 2
//         real symmetric matrix  a   h
//                                h   b
//         with lam1 > lam2 */
//
void eigen_value(float *lam1,float *lam2,float a,float b,float h)
{
*lam1=(a+b)/2.0+sqrt((a-b)*(a-b)/4.0+h*h);
*lam2=(a+b)/2.0-sqrt((a-b)*(a-b)/4.0+h*h);
}

//
// orientation() takes as input the values a,b,..,h and produces the
// angle alpha which the main axis of the corresponding ellipse makes
// with the positive x axis (range values -PI/2...PI/2)
//
float orientation(float a,float b,float c,float f,float g,float h)
{
float detnt=a*b-h*h;
float num2=a*f*f-2*h*f*g+b*g*g;
float d=num2/detnt-c;
float lam1=0.0,lam2=0.0;
float u1,u2,v1,v2,norm1,norm2;

eigen_value(&lam1,&lam2,a/d,b/d,h/d);

/*minor axis*/
norm1=sqrt(h*h/d/d+(lam1-a/d)*(lam1-a/d));
u1=h/d/norm1;
v1=(lam1-a/d)/norm1;

/*major axis*/
norm2=sqrt(h/d*h/d+(lam2-a/d)*(lam2-a/d));
u2=h/norm2/d;
v2=(lam2-a/d)/norm2;

return (atan(v2/u2));
}

//
// major_minor() takes as input the values a,b,..,h and produces the
// lengths of the major and minor axes of the corresponding ellipse
//
void major_minor(float *major,float *minor,float a,float b,float c,float f,float g,float h)
{
float detnt=a*b-h*h;
float num2=a*f*f-2*h*f*g+b*g*g;
float d=num2/detnt-c;
float lam1=0.0,lam2=0.0;

eigen_value(&lam1,&lam2,a/d,b/d,h/d);
*minor=1.0/sqrt(lam1);
*major=1.0/sqrt(lam2);
}

//
// eccen() takes as input the values a,b,..,h and produces the
// eccentricity of the corresponding ellipse
//
float eccen(float a,float b,float c,float f,float g,float h)
{
float detnt=a*b-h*h;
float num2=a*f*f-2*h*f*g+b*g*g;
float d=num2/detnt-c;
float lam1=0.0,lam2=0.0;

eigen_value(&lam1,&lam2,a/d,b/d,h/d);
return (1.0/sqrt(lam2))/(1.0/sqrt(lam1));
}
